<?php

App::uses('AppModel', 'Model');
App::uses('BlowfishPasswordHasher', 'Controller/Component/Auth');

class Link extends AppModel
{

    public $hasMany   = array( 'Stat' );
    public $belongsTo = array( 'User', 'Bundle' );
    public $validate  = array(
        'long_url'         => array(
            'notBlank'               => array(
                'on'       => 'create',
                'rule'     => 'notBlank',
                'required' => true
            ),
            'url'                    => array(
                //'on' => 'create',
                'rule'    => 'url',
                'message' => 'Please supply a valid URL address.'
            ),
            'checkBlacklistedDomain' => array(
                'on'      => 'create',
                'rule'    => 'checkBlacklistedDomain',
                'message' => 'This domain is not allowed on our system.'
            )
        ),
        'alias'            => array(
            'alphaNumeric'  => array(
                'on'         => 'create',
                'rule'       => 'alphaNumeric',
                'allowEmpty' => true,
                'message'    => 'Alias should be a alpha numeric value'
            ),
            'isUnique'      => array(
                'on'         => 'create',
                'rule'       => 'isUnique',
                'allowEmpty' => true,
                'message'    => 'Alias already exists.'
            ),
            'checkReserved' => array(
                'on'      => 'create',
                'rule'    => 'checkReserved',
                'message' => 'This Alias is a reserved word.'
            )
        ),
        'password'         => array(
            'between' => array(
                'rule'       => array( 'lengthBetween', 5, 30 ),
                //'required' => true,
                'allowEmpty' => true,
                'message'    => 'Between 5 to 30 characters'
            )
        ),
        'redirection_type' => array(
            'notBlank' => array(
                'rule' => 'notBlank',
            ),
            'inList'   => array(
                // 'rule' => array('inList', array('1', '2', '3', '4')),
                // Free only
                'rule'    => array( 'inList', array( '0', '1', '2' ) ),
                'message' => 'Choose a valid value.',
            ),
        ),
        /*
          'timer' => array(
          'naturalNumber' => array(
          'rule' => 'naturalNumber',
          'required' => true,
          'allowEmpty' => true,
          'message' => 'Please supply a valid natural number.'
          ),
          'range' => array(
          'rule' => array('range', 0, 61),
          'required' => true,
          'allowEmpty' => true,
          'message' => 'Please enter a number between 1 and 60'
          )
          ),
         */
        'bulk_links'       => array(
            'notBlank'         => array(
                'rule' => 'notBlank'
            ),
            'prepareBulkLinks' => array(
                'rule' => 'prepareBulkLinks'
            ),
        ),
        'contact_name'     => array(
            'notBlank' => array(
                'rule'    => 'notBlank',
                'message' => 'Please enter your name.'
            ),
        ),
        'contact_email'    => array(
            'notBlank' => array(
                'rule'    => 'notBlank',
                'message' => 'Please enter your email address.'
            ),
            'email'    => array(
                'rule'    => 'email',
                'message' => 'Please enter a valid email address.'
            )
        ),
        'contact_subject'  => array(
            'notBlank' => array(
                'rule'    => 'notBlank',
                'message' => 'Please enter your subject.'
            ),
        ),
        'contact_message'  => array(
            'notBlank' => array(
                'rule'    => 'notBlank',
                'message' => 'Please enter your message.'
            ),
        ),
    );

    public function prepareBulkLinks($check)
    {
        // $check will have value: array('bulk_links' => 'some-value')
        debug($check);
        return true;
    }

    public function beforeValidate($options = array())
    {
        return true;
    }

    public function beforeSave($options = array())
    {
        return true;
    }

    public function afterSave($created, $options = array())
    {
        if ($created == true) {
            $linkMeta = $this->getLinkMeta($this->data[ 'Link' ][ 'long_url' ]);

            $saved_link = array();

            $saved_link[ 'Link' ][ 'id' ]          = $this->id;
            $saved_link[ 'Link' ][ 'title' ]       = $linkMeta[ 'title' ];
            $saved_link[ 'Link' ][ 'description' ] = $linkMeta[ 'description' ];
            $saved_link[ 'Link' ][ 'image' ]       = $linkMeta[ 'og_image' ];
            $saved_link[ 'Link' ][ 'updated' ]     = false;

            $this->save($saved_link);
        }
    }

    public function originalLinkExist($long_url, $user_id)
    {
        $link = $this->find('first', array(
            'conditions' => array(
                'Link.long_url' => $long_url,
                'Link.user_id'  => $user_id
            )
        ));
        return $link;
    }

    public function getLinkMeta($long_url)
    {

        // CakePHP native behaviour
        /*
        App::uses( 'HttpSocket', 'Network/Http' );
        $HttpSocket = new HttpSocket();
        
        try {
            $response = $HttpSocket->get( $long_url , array(), array(
                'redirect' => false,
                'header'   => array(
                    'User-Agent' => env( 'HTTP_USER_AGENT' ) // Prevent old browsers problem.
                )
            ));
            $content = $response->body;
        } catch ( \Exception $ex ) {
            //pr( $ex->getMessage() );
            $content = '';
        }
        */
        
        // Prevent old browsers problem.
        // http://phpave.com/specifying-a-user-agent-when-using-file_get_contents/
        $options = array( 'http' => array( 'user_agent' => env('HTTP_USER_AGENT'), 'follow_location' => 1 ) );
        $context = stream_context_create($options);
        $content = @file_get_contents($long_url, false, $context);

        if (!empty($content)) {
            $doc   = new \DOMDocument();
            // UTF-8 Encoding Fix
            // http://www.php.net/manual/en/domdocument.loadhtml.php#95251
            @$doc->loadHTML('<?xml encoding="UTF-8">' . $content);
            $nodes = $doc->getElementsByTagName('title');

            //get and display what you need:
            if ($nodes->item(0)->nodeValue) {
                $title = $nodes->item(0)->nodeValue;
            } else {
                $title = '';
            }

            $metas       = $doc->getElementsByTagName('meta');
            $description = '';
            $og_title    = '';
            $og_image    = '';

            for ($i = 0; $i < $metas->length; $i++) {
                $meta = $metas->item($i);
                if ($meta->getAttribute('name') == 'description') {
                    $description = $meta->getAttribute('content');
                }

                // og:description
                if ($meta->getAttribute('property') == 'og:description') {
                    $og_description = $meta->getAttribute('content');
                }

                // og:title
                if ($meta->getAttribute('property') == 'og:title') {
                    $og_title = $meta->getAttribute('content');
                }
                //<meta property="og:image" content="http://www.ahram-canada.com/wp-content/uploads/2014/02/صور-المصريين.png" />
                if ($meta->getAttribute('property') == 'og:image') {
                    $og_image = $meta->getAttribute('content');
                }
            }

            if (!empty($og_title)) {
                $title = $og_title;
            }

            if (!empty($og_description)) {
                $description = $og_description;
            }

            /*
              if (empty($og_image)) {
              $og_image = $og_image;
              }
             */

            $linkMeta = array(
                'title'       => $this->cleanMeta($title),
                'description' => $this->cleanMeta($description),
                'og_image'    => $this->cleanMeta($og_image)
            );
        } else {
            $linkMeta = array(
                'title'       => '',
                'description' => '',
                'og_image'    => ''
            );
        }
        return $linkMeta;
    }

    public function cleanMeta($meta)
    {
        $meta = strip_tags($meta);
        $meta = preg_replace("/\r|\n/", "", $meta);
        return $meta;
    }

    public function checkImage($imagUrl)
    {
        if (@getimagesize($imagUrl)) {
            return 'true';
        } else {
            return 'false';
        }
    }

    public function alphaNumericDashUnderscore($check)
    {
        // $data array is passed using the form field name as the key
        // have to extract the value to make the function generic
        $value = array_values($check);
        $value = $value[ 0 ];

        return preg_match('|^[0-9a-zA-Z_-]*$|', $value);
    }

    public function checkLongURL()
    {
        if (Validation::url($this->data[ $this->alias ][ 'long_url' ], true)) {
            return true;
        } elseif (Validation::email($this->data[ $this->alias ][ 'long_url' ], true)) {
            return true;
        } else {
            return false;
        }
    }

    public function checkBlacklistedDomain($check)
    {

        // http://book.cakephp.org/2.0/en/models/data-validation.html#custom-validation-rules
        // $check will have value: array('field' => 'some-value')
        // Another method
        // $data array is passed using the form field name as the key
        // have to extract the value to make the function generic
        // $value = $check['username'];
        $value               = array_values($check);
        $blacklisted_domains = explode(',', Configure::read('Option.blacklisted_domains'));

        $blacklisted_domains[] = env('HTTP_HOST');

        $main_domain = parse_url($value[ 0 ], PHP_URL_HOST);

        if (in_array(strtolower($main_domain), $blacklisted_domains)) {
            return false;
        }
        return true;
    }

    public function isUniqueShortLink($check)
    {
        $alias_count = $this->find('count', array(
            'conditions' => array(
                //'Link.alias' => $this->data[$this->alias]['alias'],
                'Link.alias' => $check[ 'alias' ]
            )
        ));
        if ($alias_count > 0) {
            return false;
        } else {
            return true;
        }
    }

    //http://blog.justni.com/creating-a-short-url-service-using-php-and-mysql/
    public function generateurl($numAlpha)
    {
        $listAlpha   = 'abcdefghijklmnopqrstuvwxyz0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $generateurl = '';
        $i           = 0;
        while ($i < $numAlpha) {
            $random = mt_rand(0, strlen($listAlpha) - 1);
            $generateurl .= $listAlpha{$random};
            $i      = $i + 1;
        }
        return $generateurl;
    }

    function checkReserved($check)
    {
        // http://book.cakephp.org/2.0/en/models/data-validation.html#custom-validation-rules
        // $check will have value: array('field' => 'some-value')
        // Another method
        // $data array is passed using the form field name as the key
        // have to extract the value to make the function generic
        // $value = $check['username'];
        $value            = array_values($check);
        $reserved_aliases = explode(',', Configure::read('Option.reserved_aliases'));

        if (in_array($value[ 0 ], $reserved_aliases)) {
            return false;
        }
        return true;
    }

    function checkReservedAuto($keyword)
    {
        $reserved_aliases = explode(',', Configure::read('Option.reserved_aliases'));
        if (in_array($keyword, $reserved_aliases)) {
            return true;
        }
        return false;
    }

    function geturl()
    {
        do {
            $numAlpha = rand(4, 6);
            $out      = $this->generateurl($numAlpha);
            while ($this->checkReservedAuto($out)) {
                $out = $this->generateurl($numAlpha);
            }
            $alias_count = $this->find('count', array(
                'conditions' => array(
                    'Link.alias' => $out
                )
            ));
        } while ($alias_count > 0);
        return $out;
    }

    public function isOwnedBy($link_id, $user_id)
    {
        return $this->field('id', array( 'id' => $link_id, 'user_id' => $user_id )) === $link_id;
    }

    public function afterDelete()
    {
        App::uses('ClassRegistry', 'Utility');
        $statModel = ClassRegistry::init('Stat');
        $statModel->deleteAll(array( 'Stat.link_id' => $this->id ), false);

        $userModel = ClassRegistry::init('User');
        $userModel->updateAll(array( 'User.links' => 'User.links-1' ), array( 'User.id' => $this->userId ));
    }
}
